#include <fstream>
#include <iostream>
#include <boost/program_options.hpp>

#include "crypto.h"

static std::string sockname{"/var/spool/postfix/private/tempmail2"};
static std::string gensockname{"/var/run/tempmail2/generate"};
static std::string configfile{"/etc/postfix/tempmail2.conf"};

bool gDebug = false;
bool gCheckLocalpartSize = true;
bool gLowercase = false;
#if defined MAYBE_FIXED_IV
bool gNoRandomIv = false;
#endif
#if defined DEFAULT_BASE58
encode_method gEncode = encode_method::base_58;
#else
encode_method gEncode = encode_method::base_32;
#endif

void serve(const std::string& sockname);
void serve_gen(const std::string& sockname);
static int generate(const std::string& mail, int days = 3);
extern int generate_connect(const std::string& sockname, const std::string& mail, int days = 3);

static int MAX_SIZE = 64;

int main(int argc, char** argv) {
  namespace po = boost::program_options;
  po::options_description desc("Allowed options");
  desc.add_options()
    ("help,h", "describe arguments")
    ("debug,d", "enable debug output")
    ("config,c", po::value<std::string>(), "configuration file path")
    ("socket,s", po::value<std::string>(), "socket path")
    ("socket-gen,S", po::value<std::string>(), "generation socket path")
    ("encoding", po::value<std::string>(), "binary encoding method")
    ("generate,g", po::value<std::string>(), "generate a new address")
    ("generate-connect,G", po::value<std::string>(),
     "generate a new address via socket connection")
    ("expiry,e", po::value<int>(), "expiry for generate option (days)")
    ("ignore-size-limit", po::value<bool>(), "disable localpart size check")
    ("lower,l", "lowercase email in generation")
    #if defined MAYBE_FIXED_IV
    ("noiv,n", "disable random IV")
    #endif
    ("test", "run tests");

  po::options_description config("Configuration");
  config.add_options()
    ("key", po::value<std::string>(), "key to use (will be hashed)")
    ("debug", "enable debug output")
    ("max-size", po::value<int>(), "max size of email plaintext")
    ("socket-gen", po::value<std::string>(), "generation socket path")
    ("encoding", po::value<std::string>(), "binary encoding method")
    ("ignore-size-limit", po::value<bool>(), "disable localpart size check")
    ("lowercase", po::value<bool>(), "lowercase email in generation")
    #if defined MAYBE_FIXED_IV
    ("no-iv", po::value<bool>(), "disable random IV")
    #endif
    ("socket", po::value<std::string>(), "socket path");

  po::variables_map vm;
  po::store(po::parse_command_line(argc, argv, desc), vm);
  po::notify(vm);

  if (vm.count("config"))
    configfile = vm["config"].as<std::string>();

  if (vm.count("help")) {
    std::cout << "Usage: " << argv[0] << desc << std::endl;
    return 0;
  }

  std::ifstream _conf{configfile};
  if (_conf.good()) {
    po::store(po::parse_config_file(_conf, config), vm);
    po::notify(vm);
  }
  else {
    if (vm.count("config")) {
      std::cerr << "Cannot open configuartion file " << configfile << std::endl;
      return 1;
    }
  }

  if (vm.count("debug")) {
    std::cerr << "Enabling debug mode" << std::endl;
    gDebug = true;
  }

  if (vm.count("ignore-size-limit")) {
    gCheckLocalpartSize = !vm["ignore-size-limit"].as<bool>();
  }

  if (vm.count("no-iv")) {
    gNoRandomIv = vm["no-iv"].as<bool>();
    if (gDebug) std::cerr << "Disabling random IV generation" << std::endl;
  }

  if (vm.count("noiv")) {
    gNoRandomIv = true;
    if (gDebug) std::cerr << "Disabling random IV generation" << std::endl;
  }

  if (vm.count("lowercase")) {
    gLowercase = vm["lowercase"].as<bool>();
    if (gDebug) std::cerr << "Lowercase e-mail generation" << std::endl;
  }

  if (vm.count("lower")) {
    gLowercase = true;
    if (gDebug) std::cerr << "Lowercase e-mail generation" << std::endl;
  }

  if (vm.count("encoding")) {
    std::string meth = vm["encoding"].as<std::string>();
    if (gDebug) std::cerr << "Set binary encoding to " << meth << std::endl;
    if (meth == "base32")
      gEncode = encode_method::base_32;
    else if (meth == "base58") {
      gEncode =encode_method::base_58;
      if (gLowercase) {
	std::cerr << "Cannot force lowercase when using base58" << std::endl;
	return 1;
      }
    }
    else {
      std::cerr << "Unknown encoding method: " << meth << std::endl;
      return 1;
    }
  }

  if (vm.count("max-size")) {
    MAX_SIZE = vm["max-size"].as<int>();
    if (gDebug) std::cerr << "Setting maximum size to " << MAX_SIZE
                          << std::endl;
  }

  init_crypto(vm.count("key") ? vm["key"].as<std::string>() : "");

  if (vm.count("test")) {
    test_crypto();
    return 0;
  }

  if (vm.count("generate")) {
    std::string mail = vm["generate"].as<std::string>();
    if (vm.count("expiry"))
      return generate(mail, vm["expiry"].as<int>());
    else
      return generate(mail);
  }

  if (vm.count("socket"))
    sockname = vm["socket"].as<std::string>();
  if (vm.count("socket-gen"))
    gensockname = vm["socket-gen"].as<std::string>();

  if (vm.count("generate-connect")) {
    std::string mail = vm["generate-connect"].as<std::string>();
    if (vm.count("expiry"))
      return generate_connect(gensockname, mail, vm["expiry"].as<int>());
    else
      return generate_connect(gensockname, mail);
  }

  if (!gensockname.empty())
    serve_gen(gensockname);
  serve(sockname);

  return EXIT_SUCCESS;
}

static std::string __generate(const std::string& mail, int days
                              #if defined MAYBE_FIXED_IV
                              , bool aNoIv = false
                              #endif
  ) {
  auto pos = mail.find('@');
  if (pos == std::string::npos)
    throw std::runtime_error{"Missing @ in mail"};
  std::string username = mail.substr(0, pos);
  std::string domain = mail.substr(pos + 1);
  if (username.size() > MAX_SIZE)
    throw std::runtime_error{"Username too long (>" +
        std::to_string(MAX_SIZE) + ")"};
  if (domain.size() > MAX_SIZE)
    throw std::runtime_error{"Domain too long (>" +
        std::to_string(MAX_SIZE) + ")"};
  auto exp = std::chrono::system_clock::now() + std::chrono::hours(24 * days);
  if (gDebug)
    std::cerr << "generate for " << username << "@" << domain
              << " until " << date2s(exp) << std::endl;
  #if defined MAYBE_FIXED_IV
  return generate_address(username, exp, aNoIv) + "@" + domain;
  #else
  return generate_address(username, exp) + "@" + domain;
  #endif
}

std::string _generate(const std::string& mail, int days) {
  return __generate(mail, days);
}
std::string _generate_noiv(const std::string& mail, int days) {
  return __generate(mail, days, true);
}

int generate(const std::string& mail, int days) {
  try {
    std::cout << _generate(mail, days) << std::endl;
  } catch (std::exception& ex) {
    std::cerr << ex.what() << std::endl;
    return EXIT_FAILURE;
  }
  return EXIT_SUCCESS;
}
